import os

from qgis.PyQt.QtWidgets import (
    QDialog,
    QDialogButtonBox,
    QFileDialog,
    QGridLayout,
    QGroupBox,
    QLabel,
    QLineEdit,
    QMessageBox,
    QPushButton,
    QVBoxLayout,
)

from rana_qgis_plugin.constant import PLUGIN_NAME
from rana_qgis_plugin.utils import is_writable
from rana_qgis_plugin.utils_api import get_frontend_settings
from rana_qgis_plugin.utils_settings import (
    base_url,
    cognito_client_id,
    cognito_client_id_native,
    hcc_working_dir,
    rana_cache_dir,
    set_base_url,
    set_cognito_client_id,
    set_cognito_client_id_native,
    set_hcc_working_dir,
    set_rana_cache_dir,
)


class SettingsDialog(QDialog):
    def __init__(self, parent):
        super().__init__(parent)
        self.setWindowTitle(PLUGIN_NAME)
        self.setMinimumWidth(400)
        layout = QVBoxLayout(self)
        self.setLayout(layout)

        auth_group = QGroupBox("Authentication", self)
        auth_group.setLayout(QGridLayout())

        # Set up GUI and populate with settings
        auth_group.layout().addWidget(QLabel("Backend URL"), 0, 0)
        self.url_lineedit = QLineEdit(base_url(), auth_group)
        auth_group.layout().addWidget(self.url_lineedit, 0, 1)

        layout.addWidget(auth_group)

        sim_group = QGroupBox("Simulation", self)
        sim_group.setLayout(QGridLayout())
        sim_group.layout().addWidget(QLabel("Working directory"), 0, 0)
        self.working_dir_le = QLineEdit(hcc_working_dir(), sim_group)
        sim_group.layout().addWidget(self.working_dir_le, 0, 1)
        workdir_browse_pb = QPushButton("Browse", sim_group)
        sim_group.layout().addWidget(workdir_browse_pb, 0, 2)
        workdir_browse_pb.clicked.connect(
            lambda: self.browse("Working Directory", self.working_dir_le)
        )

        layout.addWidget(sim_group)

        files_group = QGroupBox("File storage", self)
        files_group.setLayout(QGridLayout())
        files_group.layout().addWidget(QLabel("Cache directory"), 0, 0)
        self.cache_dir_le = QLineEdit(rana_cache_dir(), files_group)
        files_group.layout().addWidget(self.cache_dir_le, 0, 1)
        cachedir_browse_pb = QPushButton("Browse", files_group)
        files_group.layout().addWidget(cachedir_browse_pb, 0, 2)
        cachedir_browse_pb.clicked.connect(
            lambda: self.browse("Cache Directory", self.cache_dir_le)
        )
        if rana_cache_dir() is None:
            base_dir = os.path.join(os.path.expanduser("~"), "Rana")
            self.cache_dir_le.setText(base_dir)

        layout.addWidget(files_group)

        buttonBox = QDialogButtonBox(
            QDialogButtonBox.StandardButton.Ok | QDialogButtonBox.StandardButton.Cancel
        )
        buttonBox.accepted.connect(self.accept)
        buttonBox.rejected.connect(self.reject)
        layout.addWidget(buttonBox)

        self._authenticationSettingsChanged = False

    def authenticationSettingsChanged(self):
        return self._authenticationSettingsChanged

    def accept(self) -> None:
        if self.url_lineedit.text() != base_url():
            self._authenticationSettingsChanged = True
            set_base_url(self.url_lineedit.text())

            frontend_settings = get_frontend_settings()
            if (
                frontend_settings
                and frontend_settings["default_client_id"]
                and frontend_settings["native_client_id"]
            ):
                set_cognito_client_id(frontend_settings["default_client_id"])
                set_cognito_client_id_native(frontend_settings["native_client_id"])
            else:
                QMessageBox.critical(
                    self,
                    "Error",
                    "Can't fetch settings from this backend URL. Please enter a different URL and try again.",
                )
                return

        set_hcc_working_dir(self.working_dir_le.text())
        set_rana_cache_dir(self.cache_dir_le.text())

        return super().accept()

    def browse(self, dir_human_name: str, dir_box_le: QLineEdit):
        dir = QFileDialog.getExistingDirectory(
            self, f"Select {dir_human_name}", dir_box_le.text()
        )
        if dir:
            if not is_writable(dir):
                QMessageBox.warning(
                    self,
                    "Warning",
                    "Can't write to the selected location. Please select a folder to which you have write permission.",
                )
                return
            dir_box_le.setText(dir)
